<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\Organization;
use App\Models\SubscriptionPlan;
use App\Models\OrganizationSubscription;
use App\Models\Invoice;
use App\Models\Payment;
use App\Services\SubscriptionService;
use App\Services\BillingService;
use App\Services\UsageTrackingService;
use Illuminate\Foundation\Testing\RefreshDatabase;

class SubscriptionSystemTest extends TestCase
{
    use RefreshDatabase;

    protected $organization;
    protected $plan;
    protected $subscriptionService;
    protected $billingService;
    protected $usageService;

    protected function setUp(): void
    {
        parent::setUp();

        // Create test organization
        $this->organization = Organization::factory()->create([
            'name' => 'Test Organization',
            'slug' => 'test-org',
        ]);

        // Create test plan
        $this->plan = SubscriptionPlan::create([
            'name' => 'Test Plan',
            'slug' => 'test-plan',
            'description' => 'Test plan for testing',
            'price_monthly' => 10000,
            'price_annual' => 100000,
            'max_employees' => 50,
            'max_users' => 20,
            'max_departments' => 10,
            'storage_limit_gb' => 5,
            'features' => [
                'modules' => ['employees', 'attendance'],
                'api_access' => false,
            ],
        ]);

        $this->subscriptionService = app(SubscriptionService::class);
        $this->billingService = app(BillingService::class);
        $this->usageService = app(UsageTrackingService::class);
    }

    /** @test */
    public function it_can_create_subscription_with_trial()
    {
        $subscription = $this->subscriptionService->createSubscription(
            $this->organization,
            $this->plan,
            'monthly',
            true
        );

        $this->assertNotNull($subscription);
        $this->assertEquals('trial', $subscription->status);
        $this->assertEquals($this->plan->id, $subscription->plan_id);
        $this->assertNotNull($subscription->trial_ends_at);
        $this->assertTrue($this->organization->onTrial());
    }

    /** @test */
    public function it_can_convert_trial_to_paid()
    {
        $subscription = $this->subscriptionService->createSubscription(
            $this->organization,
            $this->plan,
            'monthly',
            true
        );

        $subscription = $this->subscriptionService->convertTrialToPaid($subscription);

        $this->assertEquals('active', $subscription->status);
        $this->assertFalse($this->organization->onTrial());
    }

    /** @test */
    public function it_can_generate_invoice()
    {
        $subscription = $this->subscriptionService->createSubscription(
            $this->organization,
            $this->plan,
            'monthly',
            false
        );

        $invoice = $this->billingService->generateSubscriptionInvoice($subscription);

        $this->assertNotNull($invoice);
        $this->assertEquals($this->organization->id, $invoice->organization_id);
        $this->assertEquals($this->plan->price_monthly, $invoice->total);
        $this->assertEquals('pending', $invoice->status);
    }

    /** @test */
    public function it_can_track_usage()
    {
        $subscription = $this->subscriptionService->createSubscription(
            $this->organization,
            $this->plan,
            'monthly',
            false
        );

        $this->usageService->recordUsage($this->organization, 'employees', 10);
        $this->usageService->recordUsage($this->organization, 'users', 5);

        $summary = $this->usageService->getUsageSummary($this->organization);

        $this->assertEquals(10, $summary['employees']['current']);
        $this->assertEquals(5, $summary['users']['current']);
        $this->assertEquals(50, $summary['employees']['limit']);
        $this->assertEquals(20, $summary['users']['limit']);
    }

    /** @test */
    public function it_enforces_quota_limits()
    {
        $subscription = $this->subscriptionService->createSubscription(
            $this->organization,
            $this->plan,
            'monthly',
            false
        );

        // Record usage up to limit
        $this->usageService->recordUsage($this->organization, 'employees', 50);

        $canAdd = $this->usageService->canAdd($this->organization, 'employees');

        $this->assertFalse($canAdd);
    }

    /** @test */
    public function it_allows_adding_within_quota()
    {
        $subscription = $this->subscriptionService->createSubscription(
            $this->organization,
            $this->plan,
            'monthly',
            false
        );

        $this->usageService->recordUsage($this->organization, 'employees', 10);

        $canAdd = $this->usageService->canAdd($this->organization, 'employees');

        $this->assertTrue($canAdd);
    }

    /** @test */
    public function it_can_extend_trial()
    {
        $subscription = $this->subscriptionService->createSubscription(
            $this->organization,
            $this->plan,
            'monthly',
            true
        );

        $oldTrialEnd = $subscription->trial_ends_at;
        $subscription = $this->subscriptionService->extendTrial($subscription, 7);

        $this->assertTrue($subscription->trial_ends_at->gt($oldTrialEnd));
        $this->assertEquals(7, $oldTrialEnd->diffInDays($subscription->trial_ends_at));
    }

    /** @test */
    public function it_can_cancel_subscription()
    {
        $subscription = $this->subscriptionService->createSubscription(
            $this->organization,
            $this->plan,
            'monthly',
            false
        );

        $subscription = $this->subscriptionService->cancelSubscription($subscription);

        $this->assertEquals('cancelled', $subscription->status);
        $this->assertNotNull($subscription->ends_at);
    }

    /** @test */
    public function it_calculates_billing_summary()
    {
        $subscription = $this->subscriptionService->createSubscription(
            $this->organization,
            $this->plan,
            'monthly',
            false
        );

        // Create paid invoice
        $invoice1 = $this->billingService->generateSubscriptionInvoice($subscription);
        $invoice1->update(['status' => 'paid', 'paid_at' => now()]);

        // Create pending invoice
        $invoice2 = $this->billingService->generateSubscriptionInvoice($subscription);

        $summary = $this->billingService->getBillingSummary($this->organization);

        $this->assertEquals($this->plan->price_monthly, $summary['total_paid']);
        $this->assertEquals($this->plan->price_monthly, $summary['total_pending']);
        $this->assertEquals(2, $summary['invoice_count']);
    }
}
